package sg.edu.nyp.drmPublisher;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.Serializable;

/**
 * @author Pierre Trocme
 */
public class DMFile implements Serializable {
	private String boundaryString = null;
	private String contentType = null;
	private boolean binaryEncoding = true;
	private File contentFile;
	private boolean hasBeenWritten;

	/**
	 * Build a dm file handling Forward Lock for a specified file.
	 * @param binaryEncoding	To select binary or Base64 encoding.
	 * @param contentFile		The file to be wrapped
	 */
	public DMFile(boolean binaryEncoding, File contentFile) {
		this(contentFile);
		this.binaryEncoding = binaryEncoding;
	}
	
	/**
	 * A basic constructor.
	 * @param contentFile	the file to be managed by this drm Object.
	 */
	public DMFile(File contentFile) {
		this.contentFile = contentFile;
		String fileName = contentFile.getName();
		String extension =
			fileName.substring(fileName.lastIndexOf('.')).substring(1);
		this.contentType = new MIMEType().getMIMEContentType(extension);
		hasBeenWritten = false;
	}

	/**
	 * Write the DMFile described by this DMFile object. If the boundaryString has not been set, it is randomly generated before generating the file.
	 * @param prefix	the prefix to be append to the output filename
	 * @param path the directory where the file will be created
	 * @return	the build file.
	 * @throws IOException	if an I/O Exception occurs.
	 */
	public File generateDMFile(String prefix, String path) throws IOException {
		String dmName = contentFile.getName();
		if (boundaryString == null) {
			setRandomBoundaryString();
		}
		dmName = prefix + dmName.substring(0, dmName.lastIndexOf('.')) + ".dm";
		File output = new File(path + dmName);
		FileOutputStream fos = new FileOutputStream(output);
		DataOutputStream out = new DataOutputStream(fos);
		out.write(("--" + boundaryString + (char)0x0d +(char)0x0a ).getBytes());
		out.write(("Content-type: " + contentType + (char)0x0d +(char)0x0a ).getBytes());
		out.write(
			("Content-Transfer-Encoding: "
				+ (binaryEncoding ? "binary" : "base64")
				+ (char)0x0d +(char)0x0a )
				.getBytes());
		out.write((char)0x0d);//<CR>
		out.write((char)0x0a);//<LF>
		FileInputStream fis = new FileInputStream(contentFile);
		int nbBytes = fis.available();
		byte[] rawBytes = new byte[nbBytes];
		if (fis.read(rawBytes) != nbBytes) {
			throw new IOException();
		}
		if (binaryEncoding) {
			out.write(rawBytes);
		} else {
			out.write(
				new String(Base64Codec.encodeBase64(rawBytes)).getBytes());
		}
		out.write((char)0x0d);//<CR>
		out.write((char)0x0a);//<LF>

		out.write(("--" + boundaryString + "--").getBytes());
//		out.write((char)0x0d);//<CR>
//		out.write((char)0x0a);//<LF>
		fis.close();
		fos.close();
		hasBeenWritten = true;
		return output;
	}
	
	/**
	* Write the DMFile described by this DMFile object. If the boundaryString has not been set, it is randomly generated before generating the file.
	* @param prefix	the prefix to be append to the output filename
	* @return	the build file.
	* @throws IOException	if an I/O Exception occurs.
	*/	
	public File generateDMFile(String prefix) throws IOException {
		return generateDMFile(prefix, "");
	}

	/**
	 * Tell whether the output file will be binary encoded or not.
	 * @return	<code>true</code> if the file will be bynary encoded, else base64encoding will be used. 
	 */
	public boolean isBinaryEncoding() {
		return binaryEncoding;
	}

	/**
	 * Give the MIME content type handled by the DMFile object.
	 * @return	the MIME content type.
	 */
	public String getContentType() {
		return contentType;
	}

	/**
	 * Provide the file handled by this DMFile object.
	 * @return	the handled file.
	 */
	public File getContentFile() {
		return contentFile;
	}

	/**
	 * Set the encoding.
	 * @param b	the mode to be set
	 */
	public void setBinaryEncoding(boolean b) {
		binaryEncoding = b;
	}

	/**
	 * Set the contentType
	 * @param string	the MIME contentType to set.
	 */
	public void setContentType(String string) {
		contentType = string;
	}

	/**
	 * Giev an indication about this DMFile .dm file generation.
	 * @return	<code>true</code> if the .dm file has been at least once generated, <code>false</code> else. 
	 */
	public boolean isHasBeenWritten() {
		return hasBeenWritten;
	}

	/**
	 * Set a random boundaryString.
	 *
	 */
	public void setRandomBoundaryString() {
//		final String otherChars = "'/()+_,-.:=?";
//		//[a-zA-Z0-9'/()+_,-.:=?]
//		String toSet = new String();
//		int length = (int) Math.ceil(Math.random() * 70);
//		for (int i = 0; i < length; i++) {
//			int toAdd = (int) (Math.random() * 74);
//			if (toAdd < 26) {
//				toSet += (char) ('a' + toAdd);
//				continue;
//			}
//			if (toAdd < 52) {
//				toSet += (char) ('A' + toAdd - 26);
//				continue;
//			}
//			if (toAdd < 62) {
//				toSet += (char) ('0' + toAdd - 52);
//				continue;
//			}
//			toSet += otherChars.charAt(toAdd - 62);
//		}
//		if (!setBoundaryString(toSet.toString())) {
//			System.out.println(
//				"Unexpected error: an incorrect random boundary String has been generated.");
//			System.out.println(toSet);
//			System.out.println("The program will now exit.");
//			System.exit(1);
//		}
		String toSet = MIMEType.generateRandomBoundaryString();
		if (!setBoundaryString(toSet.toString())) {
			System.out.println(
				"Unexpected error: an incorrect random boundary String has been generated.");
			System.out.println(toSet);
			System.out.println("The program will now exit.");
			System.exit(1);	
		}	
	}

	/**
	 * @return
	 */
	public String getBoundaryString() {
		return boundaryString;
	}

	/**
	 * Set the BoundaryString. If the provided String is not RFC2046 compliant, this method does nothing.
	 * @param string	the String to set
	 * @return	<code>true</code> if the string is a RFC2046 conform boundary String, <code>false</code> in other cases.
	 */
	public boolean setBoundaryString(String string) {
		if (MIMEType.isValidBoundaryString(string)) {
			boundaryString = string;
			return true;
		} else {
			return false;
		}
	}

}
