/*
Pon : an Opening Notepad
Copyright (C) 2005 Pierre Trocm

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
/*
 * Created on 2 janv. 2005
 */
package pon;

import java.io.Serializable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Vector;
import java.util.WeakHashMap;

/**
 * A default implementation for a <code>TreeListIterator</code>.
 * @author Pierre Trocm
 * @version 0.3 - 8 janv. 2005
 */
public class DefaultTreeListIterator<E> implements Serializable,
        TreeListIterator<E>, Iterable<E> {
    private static final long serialVersionUID = -8161382002838875502L;

    private Map<E, PseudoTreeCell<E>> e2cells;

    private PseudoTreeCell<E> nextCell, prevCell;

    /**
     * Build an empty <code>DefaultTreeListIterator</code>.
     *
     */
    public DefaultTreeListIterator() {
        e2cells = new WeakHashMap<E, PseudoTreeCell<E>>();
    }

    protected class PseudoTreeCell<E> implements Serializable,
            Iterable<PseudoTreeCell<E>> {
        private static final long serialVersionUID = -4066063544565204375L;

        private PseudoTreeCell<E> myPrevCell, myNextCell;

        private List<PseudoTreeCell<E>> alternatives;

        private E object;

        public PseudoTreeCell(PseudoTreeCell<E> prevCell,
                PseudoTreeCell<E> nextCell, E object) {
            this(prevCell, nextCell, object, new Vector<PseudoTreeCell<E>>());
        }

        public PseudoTreeCell(PseudoTreeCell<E> prevCell,
                PseudoTreeCell<E> nextCell, E object,
                List<PseudoTreeCell<E>> existingList) {

            myPrevCell = prevCell;
            myNextCell = nextCell;
            this.object = object;
            alternatives = existingList;
            alternatives.add(this);
        }

        /**
         * @param e
         */
        public PseudoTreeCell(E e) {
            this(null, null, e);
        }

        public PseudoTreeCell(PseudoTreeCell<E> prevCell, E object) {
            this(prevCell, null, object);
        }

        /**
         * @return Returns the nextCell.
         */
        public PseudoTreeCell<E> getNextCell() {
            return myNextCell;
        }

        /**
         * @param nextCell
         *            The nextCell to set.
         */
        public boolean addAlternative(PseudoTreeCell<E> nextCell) {
            boolean toReturn = false;

            nextCell.setPrevCell(myPrevCell);
            toReturn = !alternatives.contains(nextCell);
            if (toReturn) {
                alternatives.add(nextCell);
            }
            return toReturn;
        }
        
        

        /**
         * @return Returns the alternatives.
         */
        public List<PseudoTreeCell<E>> getAlternatives() {
            return alternatives;
        }
        
        /**
         * @return Returns the object.
         */
        public E getObject() {
            return object;
        }

        /**
         * @param object
         *            The object to set.
         */
        public void setObject(E object) {
            this.object = object;
        }

        /**
         * @see java.lang.Iterable#iterator()
         */
        public Iterator<PseudoTreeCell<E>> iterator() {
            return alternatives.iterator();
        }

        /**
         * @return Returns the prevCell.
         */
        public PseudoTreeCell<E> getPrevCell() {
            return myPrevCell;
        }

        /**
         * @param prevCell
         *            The prevCell to set.
         */
        public void setPrevCell(PseudoTreeCell<E> prevCell) {
            myPrevCell = prevCell;
        }

        public int getAlternativesCount() {
            return alternatives.size();
        }

        public boolean hasNext() {
            return myNextCell != null;
        }

        public boolean hasPrevious() {
            return myPrevCell != null;
        }

        /**
         * @param nextCell
         *            The nextCell to set.
         */
        public void setNextCell(PseudoTreeCell<E> nextCell) {
            myNextCell = nextCell;
        }

        public PseudoTreeCell<E> getAlternative(int index) {
            try {
                return alternatives.get(index);
            } catch (ArrayIndexOutOfBoundsException e) {
                e.printStackTrace();
                throw new NoSuchElementException();
            }
        }
        
        public void setMainContinuation(E arg0){
            PseudoTreeCell<E> toMove = null;
            for(PseudoTreeCell<E> cell : alternatives){
                if(cell.getObject() == arg0){
                    toMove = cell;
                    break;
                }
            }
            if(toMove != null){
                alternatives.remove(toMove);
                alternatives.add(0, toMove);
            }
        }
    }


    /**
     * @see TreeListIterator#hasNext()
     */
    public boolean hasNext() {
        return getBranchCount() > 0;
    }

    /**
     * @see TreeListIterator#next()
     */
    public E next() {
        return next(0);
    }

    /**
     * @see TreeListIterator#hasPrevious()
     */
    public boolean hasPrevious() {
        return prevCell != null;
    }

    /**
     * @see TreeListIterator#previous()
     */
    public E previous() {
        if (!hasPrevious()) {
            throw new NoSuchElementException();
        }
        E toReturn = prevCell.getObject();
        nextCell = prevCell;
        prevCell = prevCell.getPrevCell();
        return toReturn;
    }

    /**
     * WARNING : unsupported Operation
     * @see TreeListIterator#remove()
     */
    public void remove() {
        throw new UnsupportedOperationException();
    }

    /**
     * @see TreeListIterator#set(E)
     */
    public void set(E o) {
        E oldE = nextCell.getObject();
        e2cells.remove(oldE);
        e2cells.put(o, nextCell);
        nextCell.setObject(o);
    }

    /**
     * @see TreeListIterator#add(E)
     */
    public void add(E o) {
        add(o, false);
    }

    /**
     * @see TreeListIterator#add(E, E, boolean)
     */
    private void add(E o, boolean createBranch) {
        if (createBranch) {
            PseudoTreeCell<E> toAdd = new PseudoTreeCell<E>(prevCell, o);
            
            if (nextCell != null) {
                nextCell.addAlternative(toAdd);
                for (int i = 0, n = nextCell.getAlternativesCount(); i < n; i++) {
                    toAdd.addAlternative(nextCell.getAlternative(i));
                }
            } else if (prevCell != null) {
                prevCell.setNextCell(toAdd);
            } else {
                prevCell = toAdd;
            }
            e2cells.put(o, toAdd);

        } else {
            PseudoTreeCell<E> toAdd = new PseudoTreeCell<E>(prevCell, nextCell,
                    o);
            if (prevCell != null) {
                prevCell.setNextCell(toAdd);
            }
            if (nextCell != null) {
                nextCell.setPrevCell(toAdd);
            }
            e2cells.put(o, toAdd);
            prevCell = toAdd;
        }
    }

    /**
     * @param previous
     * @param element
     * @param createBranch
     * @see TreeListIterator#add(E, E, boolean)
     */
    public void add(E previous, E element, boolean createBranch) {
        if (previous != null) {
            selectElement(previous);
        }
        add(element, createBranch);
    }

    /**
     * 
     * @return
     * @see TreeListIterator#getBranchCount()
     */
    public int getBranchCount() {
        if (nextCell != null) {
            return nextCell.getAlternativesCount();
        }

        return 0;
    }

    /**
     * @param index
     * @return
     * @see TreeListIterator#next(int)
     */
    public E next(int index) {
        if (!hasNext()) {
            throw new NoSuchElementException();
        }
        try {
            prevCell = nextCell.getAlternative(index);
            nextCell = prevCell.getNextCell();

            E toReturn = prevCell.getObject();

            return toReturn;

        } catch (IndexOutOfBoundsException e) {
            throw new NoSuchElementException();
        }
    }

    /**
     * @see pon.TreeListIterator#selectElement(pon.E)
     */
    public void selectElement(E arg0) {
        PseudoTreeCell<E> target;
        target = e2cells.get(arg0);
        if (target == null) {
            throw new NoSuchElementException();
        }
        prevCell = target;
        nextCell = target.getNextCell();
    }

    /** 
     * @see java.lang.Iterable#iterator()
     */
    public Iterator<E> iterator() {
        return e2cells.keySet().iterator();
    }

    /** 
     * @see pon.TreeListIterator#selectFirstElement()
     */
    public void selectFirstElement() {
        while (hasPrevious()) {
            previous();
        }
    }

    
    /**
     * 
     * @param arg0
     * @see TreeListIterator#setMainContinuation(E)
     */
    public void setMainContinuation(E arg0) {
        if(prevCell != null)
        prevCell.setMainContinuation(arg0);
    }
}