/*
Pon : an Opening Notepad
Copyright (C) 2005 Pierre Trocm

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
 * Created on 1 janv. 2005
 */
package pon;

import java.io.IOException;

import ictk.boardgame.Board;
import ictk.boardgame.chess.ChessGame;
import ictk.boardgame.chess.Square;
import ictk.boardgame.chess.io.FEN;

/**
 * A set of methods for handling FEN in an easier way than <code>import ictk.boardgame.chess.io.FEN;
</code>.
 * @author Pierre Trocm
 * @version 0.1 - 1 janv. 2005
 * @see ictk.boardgame.chess.io.FEN
 */
public class FENHandler {
    private static FEN fen = new FEN();
    
    private int moveNumber, pawnMoves;

    private boolean sideToMove;

    private byte castleFlag;

    private Square enPassant;

    private Board board;

    /**
     * Build a Fen Handler objet from an Fen String. Its goal is to give fas iifrmations about the elemnts contained.
     * @param fen the initiating fen.
     * @throws IOException if getBoard(fen) throws an exception
     * @see FENHandler#getBoard(String)
     */
    public FENHandler(String fen) throws IOException {
        board = getBoard(fen);
        String[] tokens = fen.split(" ");
        sideToMove = getSideToMove(tokens);

        String castleString = tokens[2];
        castleFlag = 0;
        if (castleString.contains("K")) {
            castleFlag |= 0x08;
        }
        if (castleString.contains("Q")) {
            castleFlag |= 0x04;
        }
        if (castleString.contains("k")) {
            castleFlag |= 0x02;
        }
        if (castleString.contains("q")) {
            castleFlag |= 0x01;
        }

        String enPassantString = tokens[3];
        if (enPassantString.equals("-")) {
            enPassant = null;
        } else {
            enPassant = new Square(enPassantString.charAt(0), enPassantString
                    .charAt(1));
        }

        pawnMoves = Integer.parseInt(tokens[4]);

        moveNumber = getMoveNumber(tokens);
    }

    

    /**
     * Tell whether it is white or black turn.
     * @param fen the fen to read.
     * @return ChessGame.WHITE or ChessGame.BLACK
     */
    public static boolean getSideToMove(String fen){
        return getSideToMove(fen.split(" "));
    }
    

    /**
     * Give the fen sring associated with a board. The call give the same result as new ictk.FEN().boardToString(board), but avoid the creation of a new <code>FEN </code> object.
     * @param board the board to represent as a String
     * @return a String representation of the input argument
     * @see FEN#boardToString(ictk.boardgame.Board)
     */
    public static String getFen(Board board){
        return fen.boardToString(board);
    }
    
    /**
     * @param tokens
     * @return
     */
    protected static boolean getSideToMove(String[] tokens) {
        return tokens[1].contains("w") ? ChessGame.WHITE : ChessGame.BLACK;
    }

    public static Board getBoard(String fen) throws IOException{
        return FENHandler.fen.stringToBoard(fen);
    }


    public static int getMoveNumber(String fen) {
        String[] tokens = fen.split(" ");
        return getMoveNumber(tokens);
    }

    protected static int getMoveNumber(String[] tokens) {
        return Integer.parseInt(tokens[5]);
    }

    /**
     * @return Returns the board.
     */
    public Board getBoard() {
        return board;
    }

    /**
     * @return Returns the castleFlag.
     */
    public byte getCastleFlag() {
        return castleFlag;
    }

    /**
     * @return Returns the enPassant.
     */
    public Square getEnPassant() {
        return enPassant;
    }

    /**
     * @return Returns the moveNumber.
     */
    public int getMoveNumber() {
        return moveNumber;
    }

    /**
     * @return Returns the pawnMoves.
     */
    public int getPawnMoves() {
        return pawnMoves;
    }

    /**
     * @return Returns the sideToMove.
     */
    public boolean getSideToMove() {
        return sideToMove;
    }
}