/*
 Pon : an Opening Notepad
 Copyright (C) 2005 Pierre Trocm

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package pon;

import ictk.boardgame.Board;
import ictk.boardgame.IllegalMoveException;
import ictk.boardgame.OutOfTurnException;
import ictk.boardgame.chess.AmbiguousChessMoveException;
import ictk.boardgame.chess.Bishop;
import ictk.boardgame.chess.ChessBoard;
import ictk.boardgame.chess.ChessMove;
import ictk.boardgame.chess.ChessPiece;
import ictk.boardgame.chess.King;
import ictk.boardgame.chess.Knight;
import ictk.boardgame.chess.Pawn;
import ictk.boardgame.chess.Queen;
import ictk.boardgame.chess.Rook;
import ictk.boardgame.chess.Square;
import ictk.boardgame.chess.io.FEN;
import ictk.boardgame.chess.io.SAN;

import java.awt.Container;
import java.awt.Cursor;
import java.awt.Dimension;
import java.awt.Image;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;

import javax.swing.BoxLayout;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JSeparator;
import javax.swing.JSplitPane;
import javax.swing.JTextField;
import javax.swing.JTree;
import javax.swing.SwingConstants;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.filechooser.FileFilter;
import javax.swing.tree.DefaultMutableTreeNode;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

import com.infosys.closeandmaxtabbedpane.CloseAndMaxTabbedPane;
import com.infosys.closeandmaxtabbedpane.CloseListener;
import com.infosys.closeandmaxtabbedpane.DoubleClickListener;

/*
 * GUI.java
 * 
 * Created on 30 novembre 2004, 15:49
 */

/**
 * 
 * @author Pierre Trocm
 * @version 0.9 - 9 jan. 2005
 */

public class GUI extends JFrame {

    private static final long serialVersionUID = 5998330575892562544L;

    /** Creates new form GUI */
    public GUI() {
        super("PON : an Opening Notepad");
        initComponents();
        linker = new Linker();
    }

    public void fillTree(ChapitreMember currentMember,
            DefaultMutableTreeNode from) {
        fillTree(currentMember, from, true);
    }

    public void fillTree(ChapitreMember currentMember,
            DefaultMutableTreeNode from, boolean includePages) {
        if (includePages || !(currentMember instanceof Page)) {

            DefaultMutableTreeNode node = new DefaultMutableTreeNode(
                    currentMember);
            from.add(node);

            for (ChapitreMember next : currentMember) {
                fillTree(next, node, includePages);
            }
        }
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    private void initComponents() {
        jSplitPane1 = new JSplitPane();
        rootNode = new DefaultMutableTreeNode("Sommaire");
        jTree1 = new JTree(rootNode);

        setPreferredSize(new Dimension(800, 600));

        jTree1.addTreeSelectionListener(new TreeSelectionListener() {

            public void valueChanged(TreeSelectionEvent e) {
                DefaultMutableTreeNode node = (DefaultMutableTreeNode) jTree1
                        .getLastSelectedPathComponent();

                if (node == null)
                    return;

                if (node.getUserObject() instanceof Page) {
                    loadPage((Page) node.getUserObject());
                }
            }

        });

        jTabbedPane1 = new CloseAndMaxTabbedPane(false);
        jTabbedPane1.setMaxIcon(false);

        jTabbedPane1.addDoubleClickListener(new DoubleClickListener() {

            public void doubleClickOperation(MouseEvent e) {
                int index = jTabbedPane1.getSelectedIndex();
                jTabbedPane1.detachTab(index);
            }
        });

        //LATER on ne peut pas loader un node selected

        jTabbedPane1.addCloseListener(new CloseListener() {

            public void closeOperation(MouseEvent e) {
                PageView toUnload = (PageView) jTabbedPane1
                        .getSelectedComponent();
                unloadPage(toUnload.getPage());
            }
        });

        jMenuBar1 = new JMenuBar();
        fichier = new JMenu("Fichier");
        editer = new JMenu("Editer");
        organiser = new JMenu("Organiser");

        newPage = new JMenuItem("Nouvelle Page");
        load = new JMenuItem("Charger");
        save = new JMenuItem("Sauvegarder");
        newChapter = new JMenuItem("Nouveau Chapitre");
        newLinker = new JMenuItem("Nouveau carnet");
        movePage = new JMenuItem("Dplacer une page");
        moveChapter = new JMenuItem("Dplacer un chapitre");
        renameChapter = new JMenuItem("Renommer un chapitre");
        renamePage = new JMenuItem("Renommer une page");
        deleteChapter = new JMenuItem("Supprimer un chapitre");
        deletePage = new JMenuItem("Supprimer une page");

        setDefaultLookAndFeelDecorated(false);

        setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

        jSplitPane1.setLeftComponent(jTree1);

        jSplitPane1.setRightComponent(jTabbedPane1);

        getContentPane().add(jSplitPane1, java.awt.BorderLayout.CENTER);

        load.addActionListener(new ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                importer(evt);
            }
        });

        moveChapter.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                Chapitre toMove, destination;
                toMove = showChapterSelector("Chapitre  Dplacer");
                if (toMove == null) {
                    return;
                }
                destination = showChapterSelector("Dstination");
                if (destination == null) {
                    return;
                }
                destination.add(toMove);
                resetTree();
            }
        });

        movePage.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                Chapitre destination;
                Page toMove;
                toMove = showPageSelector("Page  Dplacer");
                if (toMove == null) {
                    return;
                }
                destination = showChapterSelector("Dstination");
                if (destination == null) {
                    return;
                }
                destination.add(toMove);
                resetTree();
            }
        });

        newPage.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                showPageCreator();
            }
        });

        newChapter.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                Chapitre toAdd;
                Chapitre chapitre;
                String nom = JOptionPane.showInputDialog(GUI.this,
                        "Nom de la page : ", "Nouvelle Page...",
                        JOptionPane.QUESTION_MESSAGE);
                toAdd = new Chapitre(nom);

                chapitre = showChapterSelector();
                if (chapitre != null) {
                    chapitre.add(toAdd);
                    resetTree(linker.getSommaire());
                }
            }
        });

        newLinker.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                Linker newLinker = new Linker();
                loadLinker(newLinker);
            }
        });

        save.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                sauver();
            }
        });

        renameChapter.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                Chapitre chapitre = showChapterSelector();
                if (chapitre == null) {
                    return;
                }
                String nom = JOptionPane.showInputDialog(GUI.this,
                        "Nom du Chapitre: ", "Renommer un chapitre ...",
                        JOptionPane.QUESTION_MESSAGE);
                chapitre.setName(nom);
                resetTree();

            }
        });

        renamePage.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                Page page = showPageSelector("Slectionner une page");
                if (page == null) {
                    return;
                }
                String nom = JOptionPane.showInputDialog(GUI.this,
                        "Nom de la page : ", "Renommer une page ...",
                        JOptionPane.QUESTION_MESSAGE);
                unloadPage(page);
                page.setName(nom);
                resetTree();

            }
        });

        deletePage.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                //FIXME on ne peux pas supprimer un mallon sans suceseurs
                //TODO les transpositions sont dtctes un coup trop tot
                Page toDelete = showPageSelector("Choisissez une page");
                if (toDelete != null) {
                    unloadPage(toDelete);
                    toDelete.getChapitre().remove(toDelete);
                    linker.deleteMaillon(toDelete.getFen(), toDelete
                            .getFirstMaillon(), toDelete);
                    resetTree();
                }
            }
        });

        deleteChapter.addActionListener(new ActionListener() {

            public void actionPerformed(ActionEvent e) {
                Chapitre toDelete = showChapterSelector();
                if (toDelete != null && toDelete != linker.getSommaire()) {
                    for (ChapitreMember member : toDelete) {
                        member.setChapitre(toDelete.getChapitre());
                    }
                    toDelete.getChapitre().remove(toDelete);
                    resetTree();
                }
            }
        });

        fichier.add(load);
        fichier.add(save);
        fichier.add(newLinker);

        organiser.add(newPage);
        organiser.add(newChapter);
        organiser.add(moveChapter);
        organiser.add(movePage);
        organiser.add(renamePage);
        organiser.add(renameChapter);
        organiser.add(deletePage);
        organiser.add(deleteChapter);

        jMenuBar1.add(fichier);
        jMenuBar1.add(editer);
        jMenuBar1.add(organiser);

        setJMenuBar(jMenuBar1);

        pack();
    }

    protected Page showPageCreator() {
        return showPageCreator(Page.StartFEN, linker.getSommaire());
    }

    /**
     * 
     * @param fen
     */
    protected Page showPageCreator(String fen, Chapitre chapitre) {
        Page toAdd = null;
        JTextField nameField = new JTextField("Page Sans Nom");
        JTextField fenField = new JTextField(fen);
        JLabel chpLabel = new JLabel("Chapitre : " + chapitre);
        Object[] message = { "Nom de la page : ", nameField, chpLabel,
                "Position de dpart : ", fenField };
        Object[] options = { "OK", "Annuler", "Choisir le Chapitre..." };
        int choice = JOptionPane.showOptionDialog(this, message,
                "Nouvelle Page...", JOptionPane.OK_CANCEL_OPTION,
                JOptionPane.QUESTION_MESSAGE, null, options, options[0]);
        fen = fenField.getText();

        try {
            FENHandler.getBoard(fen);
        } catch (Exception e) {
            System.out.println("Fen incorrecte.");
            fen = Page.StartFEN;
        }
        switch (choice) {
        case 2:
            Chapitre chapitreTmp = showChapterSelector();
            if (chapitreTmp != null) {
                chapitre = chapitreTmp;
            }
        case 0:
            toAdd = new Page(nameField.getText(), fen);

            if (chapitre != null) {
                chapitre.add(toAdd);
                resetTree(linker.getSommaire());
            }
            break;
        case 1:
            break;
        default:
            break;

        }
        return toAdd;
    }

    private void sauver() {
        int choice;
        JFileChooser chooser = new JFileChooser();
        chooser.setFileFilter(new FileFilter() {

            public boolean accept(File f) {
                return f.getName().endsWith(".pon");
            }

            public String getDescription() {
                return "Pon files (.pon)";
            }
        });
        choice = chooser.showSaveDialog(this);
        if (choice == JFileChooser.APPROVE_OPTION) {
            File file = chooser.getSelectedFile();
            linker.save(file);
        }
    }

    protected PageView loadPage(Page page) {
        return loadPage(page, true);
    }

    protected PageView loadPage(Page page, boolean selectPage) {
        int index = 0;
        PageView toLoad = null;
        boolean isLoaded = page.isLoaded();
        if (isLoaded) {
            for (int i = 0, n = jTabbedPane1.getTabCount(); i < n; i++) {
                if (jTabbedPane1.getTitleAt(i).equals(page.getName())) {
                    index = i;
                    break;
                    //LATER viter les parcours (revenir  loadIndex?)
                }
            }
        } else {
            try {
                //                ImageIcon closeButton = new
                // ImageIcon("pictures/closeicon.gif");
                toLoad = new PageView(page, this);

                jTabbedPane1.addTab(page.getName(), null, toLoad);
                //              jTabbedPane1.addTab(page.getName(), closeButton, toLoad);
                index = jTabbedPane1.indexOfComponent(toLoad);
                page.setLoaded(true);
                jTabbedPane1.setSelectedIndex(index);
            } catch (IOException e) {
                e.printStackTrace();
            } catch (OutOfTurnException e) {
                e.printStackTrace();
            } catch (AmbiguousChessMoveException e) {
                e.printStackTrace();
            } catch (IllegalMoveException e) {
                e.printStackTrace();
            } catch (IndexOutOfBoundsException e) {
                e.printStackTrace();
            }
        }
        if (selectPage) {
            jTabbedPane1.setSelectedIndex(index);
        }
        return toLoad;
    }

    /**
     * 
     * @param page
     * @param moveView
     * @version 0.2 - 28 Dcembre 2004
     */
    protected void loadPage(Page page, MoveView moveView, boolean selectPage) {
        PageView toLoad = loadPage(page, selectPage);
        if (moveView != null && toLoad != null) {
            toLoad.setSelectedMoveView(moveView);
        }
    }

    protected void loadPage(Page page, MoveView moveView) {
        loadPage(page, moveView, true);
    }

    protected void loadPage(Page page, String fenToLoad) {
        loadPage(page, fenToLoad, true);
    }

    protected void loadPage(Page page, String fenToLoad, boolean selectPage) {
        PageView toLoad = loadPage(page, selectPage);
        MoveView moveView = toLoad.getMoveViewFromResultingFen(fenToLoad);

        if (toLoad != null && fenToLoad != null && moveView != null) {
            toLoad.setSelectedMoveView(moveView);
        }
    }

    protected void reloadPage(Page toReload, MoveView moveView) {
        reloadPage(toReload, moveView, true);
    }

    protected void reloadPage(Page toReload, MoveView moveView,
            boolean selectPage) {
        unloadPage(toReload);
        loadPage(toReload, moveView, selectPage);
    }

    protected void reloadPage(Page toReload, String fenToSet, boolean selectPage) {
        unloadPage(toReload);
        loadPage(toReload, fenToSet, selectPage);
    }

    protected void reloadPage(Page toReload, String fenToSet) {
        reloadPage(toReload, fenToSet, true);
    }

    private void unloadPage(Page page) {
        if (page.isLoaded()) {
            for (int i = 0, n = jTabbedPane1.getTabCount(); i < n; i++) {
                if (jTabbedPane1.getTitleAt(i).equals(page.getName())) {
                    jTabbedPane1.remove(i);
                    page.setLoaded(false);
                    break;
                }
            }
        }
    }

    private void importer(java.awt.event.ActionEvent evt) {
        int choice;
        JFileChooser chooser = new JFileChooser();
        chooser.setFileFilter(new FileFilter() {

            public boolean accept(File f) {
                String name = f.getName();
                return name.endsWith(".sor") || name.endsWith(".pon");
            }

            public String getDescription() {
                return "Scid opening repertoire File (.sor), Pon Files (.pon)";
            }
        });
        //        chooser.setAcceptAllFileFilterUsed(true);
        chooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
        choice = chooser.showOpenDialog(this);
        if (choice == JFileChooser.APPROVE_OPTION) {
            File file = chooser.getSelectedFile();
            if (file.getName().endsWith(".sor")) {
                loadLinker(Linker.loadSorFile(file));
            } else if (file.getName().endsWith(".pon")) {
                try {
                    loadLinker(Linker.load(file));

                } catch (FileNotFoundException e) {
                    e.printStackTrace();
                } catch (IOException e) {
                    e.printStackTrace();
                } catch (ClassNotFoundException e) {
                    e.printStackTrace();
                } catch (ClassCastException e) {
                    System.out.println("Fichier invalide");
                }
            }
        }
    }

    private void loadLinker(Linker linker) {
        this.linker = linker;
        resetTree(linker.getSommaire());
        jTabbedPane1.removeAll();
    }

    // Variables declaration - do not modify
    private JMenuItem newPage, load, save, newChapter, newLinker, movePage,
            moveChapter, renamePage, renameChapter, deletePage, deleteChapter;

    private JMenu fichier, editer, organiser;

    private JMenuBar jMenuBar1;

    private JSplitPane jSplitPane1;

    private CloseAndMaxTabbedPane jTabbedPane1;

    private JTree jTree1;

    private DefaultMutableTreeNode rootNode;

    private Linker linker;

    // End of variables declaration

    private void resetTree() {
        resetTree(linker.getSommaire());
    }

    private void resetTree(Chapitre sommaire) {
        Container a = jTree1.getParent();
        a.remove(jTree1);
        rootNode = new DefaultMutableTreeNode(sommaire);
        jTree1 = new JTree(rootNode);
        for (ChapitreMember chapitreMember : sommaire) {
            fillTree(chapitreMember, rootNode);
        }

        jTree1.addTreeSelectionListener(new TreeSelectionListener() {

            public void valueChanged(TreeSelectionEvent e) {
                DefaultMutableTreeNode node = (DefaultMutableTreeNode) jTree1
                        .getLastSelectedPathComponent();

                if (node == null)
                    return;

                if (node.getUserObject() instanceof Page) {
                    loadPage((Page) node.getUserObject());
                }
            }

        });
        jSplitPane1.setLeftComponent(jTree1);

    }

    public Chapitre showChapterSelector() {
        return showChapterSelector("Choisissez un chapitre");
    }

    public Chapitre showChapterSelector(String title) {
        Chapitre toReturn = null;
        JDialog dialog;
        JOptionPane optionPane = new JOptionPane();
        ChapitreMember sommaire = linker.getSommaire();
        DefaultMutableTreeNode treeRoot = new DefaultMutableTreeNode(sommaire);
        JTree tree = new JTree(treeRoot);
        tree.setSelectionInterval(0, 0);
        for (ChapitreMember chapitreMember : sommaire) {
            fillTree(chapitreMember, treeRoot, false);
        }

        optionPane.setOptionType(JOptionPane.OK_CANCEL_OPTION);
        optionPane.setMessage(tree);
        expandAll(tree, true);
        dialog = optionPane.createDialog(this, title);

        dialog.setVisible(true);

        Integer selectedValue = (Integer) optionPane.getValue();
        switch (selectedValue.intValue()) {
        case JOptionPane.OK_OPTION:
            DefaultMutableTreeNode node = (DefaultMutableTreeNode) tree
                    .getLastSelectedPathComponent();
            toReturn = (Chapitre) node.getUserObject();
            break;
        case JOptionPane.CANCEL_OPTION:
        default:
            break;
        }
        return toReturn;
    }

    public Page showPageSelector(String title) {
        Page toReturn = null;
        JDialog dialog;
        JOptionPane optionPane = new JOptionPane();
        ChapitreMember sommaire = linker.getSommaire();
        DefaultMutableTreeNode treeRoot = new DefaultMutableTreeNode(sommaire);
        JTree tree = new JTree(treeRoot);
        for (ChapitreMember chapitreMember : sommaire) {
            fillTree(chapitreMember, treeRoot, true);
        }

        optionPane.setOptionType(JOptionPane.OK_CANCEL_OPTION);
        optionPane.setMessage(tree);
        expandAll(tree, true);
        dialog = optionPane.createDialog(this, title);

        dialog.setVisible(true);

        Integer selectedValue = (Integer) optionPane.getValue();
        switch (selectedValue.intValue()) {
        case JOptionPane.OK_OPTION:
            try {
                DefaultMutableTreeNode node = (DefaultMutableTreeNode) tree
                        .getLastSelectedPathComponent();
                toReturn = (Page) node.getUserObject();
            } catch (ClassCastException e) {
                e.printStackTrace();
            }
            break;

        case JOptionPane.CANCEL_OPTION:
        default:
            break;
        }
        return toReturn;
    }

    /**
     * If expand is true, expands all nodes in the tree. Otherwise, collapses
     * all nodes in the tree.
     * 
     * @param tree
     * @param expand
     * @author javaAlmanac.com
     * @see http://javaalmanac.com/egs/javax.swing.tree/ExpandAll.html
     */
    public static void expandAll(JTree tree, boolean expand) {
        TreeNode root = (TreeNode) tree.getModel().getRoot();

        // Traverse tree from root
        expandAll(tree, new TreePath(root), expand);
    }

    /**
     * 
     * @param tree
     * @param parent
     * @param expand
     * @author javaAlmanac.com
     * @see http://javaalmanac.com/egs/javax.swing.tree/ExpandAll.html
     */
    private static void expandAll(JTree tree, TreePath parent, boolean expand) {
        // Traverse children
        TreeNode node = (TreeNode) parent.getLastPathComponent();
        if (node.getChildCount() >= 0) {
            for (Enumeration e = node.children(); e.hasMoreElements();) {
                TreeNode n = (TreeNode) e.nextElement();
                TreePath path = parent.pathByAddingChild(n);
                expandAll(tree, path, expand);
            }
        }

        // Expansion or collapse must be done bottom-up
        if (expand) {
            tree.expandPath(parent);
        } else {
            tree.collapsePath(parent);
        }
    }

    /**
     * 
     * @param list
     * @param baseFen
     * @param variationLevel
     * @param fromMaillon
     * @param page
     * @throws IOException
     * @throws OutOfTurnException
     * @throws AmbiguousChessMoveException
     * @throws IllegalMoveException
     * @deprecated utiliser la version ave treeListIterator
     */
    private void getAllMoveViewFrom(List<MoveView> list, String baseFen,
            int variationLevel, Maillon fromMaillon, Page page)
            throws IOException, OutOfTurnException,
            AmbiguousChessMoveException, IllegalMoveException {
        FEN fen = new FEN();
        SAN san = new SAN();
        while (fromMaillon != null) {
            Board board = fen.stringToBoard(baseFen);
            board.playMove(san.stringToMove(board, fromMaillon.getMove()));
            baseFen = fen.boardToString(board);

            if (fromMaillon.getPage() == page) {
                list.add(new MoveView(baseFen, fromMaillon, variationLevel));
            } else {
                list.add(new MoveView(baseFen, fromMaillon, variationLevel + 1,
                        "? " + fromMaillon.getPage().getName()));
                break;
            }

            fromMaillon = linker.getMainLineFrom(baseFen);
            List<Maillon> variations = linker.getVariationsFrom(baseFen);
            for (Maillon maillon : variations) {
                getAllMoveViewFrom(list, baseFen, variationLevel + 1, maillon,
                        page);
            }
        }
    }

    protected void getAllMoveViewFrom(TreeListIterator<MoveView> list,
            String baseFen, int variationLevel, Maillon fromMaillon,
            MoveView previousMaillon, Page page) throws IOException,
            OutOfTurnException, AmbiguousChessMoveException,
            IllegalMoveException {
        //LATER dplacer apres dans linker
        FEN fen = new FEN();
        SAN san = new SAN();
        if (fromMaillon != null) {
            //Ajouter le coup li a fromMaillon
            //updater
            Board board = fen.stringToBoard(baseFen);
            board.playMove(san.stringToMove(board, fromMaillon.getMove()));
            baseFen = fen.boardToString(board);

            MoveView toAdd = fromMaillon.getPage() == page ? new MoveView(
                    baseFen, fromMaillon, variationLevel) : new MoveView(
                    baseFen, fromMaillon, variationLevel, "-> "
                            + fromMaillon.getPage());

            list.add(previousMaillon, toAdd, true);

            if (fromMaillon.getPage() != page) {
                return;
            }

            fromMaillon = linker.getMainLineFrom(baseFen);
            previousMaillon = toAdd;

            //POUR chaque coup faisant parti de ses successeurs
            //Rcursiver

            getAllMoveViewFrom(list, baseFen, variationLevel, fromMaillon,
                    previousMaillon, page);

            List<Maillon> variations = linker.getVariationsFrom(baseFen);
            for (Maillon maillon : variations) {
                getAllMoveViewFrom(list, baseFen, variationLevel + 1, maillon,
                        previousMaillon, page);
            }
        }
    }

    protected void changeLine(Maillon maillon, Page fromPage, Page toPage,
            String fromFen) {
        List<MoveView> list = new ArrayList<MoveView>();
        try {
            getAllMoveViewFrom(list, fromFen, 0, maillon, fromPage);
        } catch (OutOfTurnException e) {
            e.printStackTrace();
        } catch (AmbiguousChessMoveException e) {
            e.printStackTrace();
        } catch (IllegalMoveException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        for (MoveView moveView : list) {
            Maillon maillon2 = moveView.getMaillon();
            if (maillon2.getPage() == fromPage) {
                maillon2.setPage(toPage);
            }
        }
    }

    /**
     * 
     * @author Pierre Trocm
     * @version 0.2 - 26 dc. 2004
     */
    private class PageView extends JPanel implements MouseListener,
            ActionListener {

        private static final long serialVersionUID = 1L;

        private BoardView boardView;

        private String currentFen;

        private Square dragStartSquare;

        private MovePanel movePanel;

        private JPanel boardPanel;

        private MoveView selectedMoveView;

        private JButton startButton, backwardButton, forwardButton, endButton;

        private TreeListIterator<MoveView> treeListIterator;

        private Page page;

        public PageView(Page page, GUI gui) throws OutOfTurnException,
                AmbiguousChessMoveException, IllegalMoveException, IOException {
            super();
            setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
            boardView = new BoardView(page.getRootFen());
            boardView.addMouseListener(this);
            this.page = page;
            currentFen = page.getFen();

            movePanel = new MovePanel();
            boardPanel = new JPanel();

            boardPanel
                    .setLayout(new BoxLayout(boardPanel, BoxLayout.PAGE_AXIS));
            boardPanel.add(boardView);

            JPanel buttonsPanel = new JPanel();
            buttonsPanel.setLayout(new BoxLayout(buttonsPanel,
                    BoxLayout.LINE_AXIS));
            startButton = new JButton("|<");
            backwardButton = new JButton("<");
            forwardButton = new JButton(">");
            endButton = new JButton(">|");

            startButton.addActionListener(this);
            backwardButton.addActionListener(this);
            forwardButton.addActionListener(this);
            endButton.addActionListener(this);

            buttonsPanel.add(startButton);
            buttonsPanel.add(backwardButton);
            buttonsPanel.add(forwardButton);
            buttonsPanel.add(endButton);

            boardPanel.add(buttonsPanel);
            add(boardPanel);
            add(movePanel);

            //FAST IMPLEMENTATION
            String baseFen = page.getFen();

            FEN fen = new FEN();
            SAN san = new SAN();
            //Premier coup : on ne compte pas les variations
            Maillon mainLine = page.getFirstMaillon();
            //l'chiquier est initialis  la base Fen

            //Ajout du prefixe
            //LATER en cas d'insertion que fait-on?
            //LATER penser  permettre de dfinir le prfixe avant les coups
            // originaux
            String prefixFen = page.getPrefixFen();

            treeListIterator = new DefaultTreeListIterator<MoveView>();
            List<MoveView> moveViewPrefix = new ArrayList<MoveView>();
            for (Maillon maillon : page.getPrefix()) {
                Board board = fen.stringToBoard(prefixFen);
                board.playMove(san.stringToMove(board, maillon.getMove()));
                prefixFen = fen.boardToString(board);
                moveViewPrefix.add(new MoveView(this, prefixFen, maillon));
            }

            movePanel.add(moveViewPrefix);

            movePanel.add(new JSeparator(SwingConstants.HORIZONTAL));

            addMoveView(baseFen, 0, mainLine, page);

            treeListIterator.selectFirstElement();
            if (treeListIterator.hasNext()) {
                selectedMoveView = treeListIterator.next();
                selectedMoveView.setSelected(true);
            }
        }

        /**
         * @param view
         */
        private void setSelectedMoveView(MoveView moveView) {
            if (!movePanel.getList().contains(moveView)) {
                moveView = getMoveViewFromResultingFen(moveView
                        .getResultingFen());
            }
            //LATER trouver comment viter le contains()

            if (moveView != null) {
                currentFen = moveView.getResultingFen();
                boardView.setFen(currentFen);

                treeListIterator.selectElement(moveView);

                selectedMoveView.setSelected(false);
                selectedMoveView = moveView;
                selectedMoveView.setSelected(true);
            }
        }

        /**
         * @param resultingFen
         */
        private MoveView getMoveViewFromResultingFen(String resultingFen) {
            Iterator<MoveView> iterator = movePanel.listIterator();
            while (iterator.hasNext()) {
                MoveView moveView = iterator.next();
                if (moveView.getResultingFen().equals(resultingFen)) {
                    return moveView;
                }
            }
            return null;
        }

        private void addMoveView(String baseFen, int variationLevel,
                Maillon toAdd, Page page) throws IOException,
                OutOfTurnException, AmbiguousChessMoveException,
                IllegalMoveException {

            getAllMoveViewFrom(treeListIterator, baseFen, 0, toAdd, null, page);

            for (MoveView moveView : treeListIterator) {
                moveView.addMouseListener(this);
            }

            treeListIterator.selectFirstElement();

            movePanel.add(treeListIterator);
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.awt.event.MouseListener#mouseClicked(java.awt.event.MouseEvent)
         */
        public void mouseClicked(MouseEvent e) {
            if (e.getSource() instanceof MoveView) {
                final MoveView moveView = (MoveView) e.getSource();
                setSelectedMoveView(moveView);
                switch (e.getButton()) {
                case 1:

                    switch (e.getClickCount()) {
                    case 1:
                        break;
                    case 2:
                        loadPage(moveView.getMaillon().getPage(), moveView);
                        break;
                    }
                    break;
                case 2:
                    break;
                case 3:
                    JPopupMenu menu = new JPopupMenu();
                    JMenuItem delete,
                    moveHere,
                    newPage,
                    promote;

                    delete = new JMenuItem("Supprimer...");
                    moveHere = new JMenuItem("Dplacer dans cette page");
                    newPage = new JMenuItem("Crer une nouvelle page...");
                    promote = new JMenuItem("Promouvoir");

                    delete.addActionListener(new ActionListener() {

                        public void actionPerformed(ActionEvent e) {
                            Maillon toDelete = moveView.getMaillon();
                            if (toDelete != page.getFirstMaillon()) {
                                selectPrevMoveView();
                                String fromFen = selectedMoveView
                                        .getResultingFen();

                                linker.deleteMaillon(fromFen, toDelete, page);
                            } else {
                                String fromFen = page.getFen();
                                linker.deleteMaillon(fromFen, toDelete, page);
                                page.setFirstMaillon(null);
                            }
                            reloadPage(page, selectedMoveView);
                        }
                    });
                    //LATER ajouter les coups du prfixe au treeListIterator

                    moveHere.addActionListener(new ActionListener() {

                        public void actionPerformed(ActionEvent e) {
                            if (selectedMoveView.getMaillon().getPage() != page) {
                                Maillon fromMaillon = selectedMoveView
                                        .getMaillon();

                                selectPrevMoveView();
                                String fromFen = selectedMoveView
                                        .getResultingFen();

                                Page oldPage = moveView.getMaillon().getPage();
                                changeLine(fromMaillon, fromMaillon.getPage(),
                                        page, fromFen);

                                reloadPage(oldPage, moveView);
                                reloadPage(page, moveView);
                            }
                        }
                    });

                    newPage.addActionListener(new ActionListener() {

                        public void actionPerformed(ActionEvent e) {
                            selectPrevMoveView();
                            String fromFen = selectedMoveView.getResultingFen();
                            Maillon maillon = moveView.getMaillon();

                            Page toAdd = showPageCreator(fromFen, page
                                    .getChapitre());
                            toAdd.setFirstMaillon(maillon);

                            changeLine(maillon, page, toAdd, fromFen);
                            selectNextMoveView();

                            reloadPage(page, moveView);
                            reloadPage(toAdd, moveView);

                        }
                    });

                    //LATER ajouter un premier maillon titre / baseFen
                    //Permettrait de trouver des previous quand il n'y en a pas

                    promote.addActionListener(new ActionListener() {
                        public void actionPerformed(ActionEvent e) {
                            selectPrevMoveView();
                            List<Maillon> list = linker
                                    .getMaillonsElementsFrom(selectedMoveView
                                            .getResultingFen());
                            Maillon maillon = moveView.getMaillon();
                            list.remove(maillon);
                            list.add(0, maillon);

                            reloadPage(page, moveView);
                        }
                    });

                    menu.add(delete);
                    menu.add(moveHere);
                    menu.add(newPage);
                    menu.add(promote);

                    menu.show(movePanel, e.getX(), e.getY());
                    //TODO associer des commentaires aux coups/les afficher
                    //TODO renommer les classes en anglais
                    break;
                }
            }
        }

        /**
         * @see java.awt.event.MouseListener#mousePressed(java.awt.event.MouseEvent)
         */
        public void mousePressed(MouseEvent e) {
            if (e.getSource() == boardView
                    && (selectedMoveView == null || (selectedMoveView
                            .getResultingFen() == currentFen && selectedMoveView
                            .getPage() == page))) {
                double x, y;
                x = Math.floor(e.getX() * 8.0 / boardView.getSize().width);
                y = 8 - Math.floor(e.getY() * 8.0 / boardView.getSize().height);

                FEN fen = new FEN();
                try {
                    Board board = fen.stringToBoard(currentFen);
                    dragStartSquare = ((ChessBoard) board).getSquare(
                            (char) (x + 'a'), (char) (y + '0'));

                    ChessPiece piece = (ChessPiece) dragStartSquare.getPiece();
                    if (piece != null) {
                        String path = "w";

                        byte pieceIndex = piece.getIndex();
                        if (piece.isBlack()) {
                            pieceIndex -= ChessPiece.BLACK_OFFSET;
                            path += 'b';
                        } else {
                            path += 'w';
                        }
                        //LATER ne pas recharger tout la page, seulement le
                        // movePanel
                        switch (pieceIndex) {
                        case King.INDEX:
                            path += "k.gif";
                            break;
                        case Queen.INDEX:
                            path += "q.gif";
                            break;
                        case Pawn.INDEX:
                            path += "p.gif";
                            break;
                        case Knight.INDEX:
                            path += "n.gif";
                            break;
                        case Bishop.INDEX:
                            path += "b.gif";
                            break;
                        case Rook.INDEX:
                            path += "r.gif";
                            break;
                        }

                        Cursor cur_cust = null;
                        Toolkit tk = Toolkit.getDefaultToolkit();
                        Image img = new ImageIcon("Cursors/" + path).getImage();
                        Point p = new Point(img.getWidth(null) / 2, img
                                .getHeight(null) / 2); // This is the hot-spot
                        // in
                        // the cursor
                        try {
                            cur_cust = tk.createCustomCursor(img, p,
                                    "Cursor ID");
                        } catch (IndexOutOfBoundsException ioobe) {
                            System.out.println(ioobe);
                        }

                        setCursor(cur_cust);

                        dragStartSquare.setPiece(null);
                        boardView.setFen(fen.boardToString(board));
                        dragStartSquare.setPiece(piece);
                    } else {
                        dragStartSquare = null;
                    }
                } catch (IOException e1) {
                    e1.printStackTrace();
                }

            }
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.awt.event.MouseListener#mouseReleased(java.awt.event.MouseEvent)
         */
        public void mouseReleased(MouseEvent e) {
            if (e.getSource() == boardView && dragStartSquare != null) {
                setCursor(Cursor.getDefaultCursor());
                if (boardView.contains(e.getX(), e.getY())) {
                    try {
                        double x, y;
                        FEN fen = new FEN();
                        SAN san = new SAN();

                        x = Math.floor(e.getX() * 8.0
                                / boardView.getSize().width);
                        y = 8 - Math.floor(e.getY() * 8.0
                                / boardView.getSize().height);
                        Square endSquare = new Square((char) (x + 'a'),
                                (char) (y + '0'));

                        ChessBoard board = (ChessBoard) fen
                                .stringToBoard(currentFen);

                        ChessMove move;
                        ChessPiece piece = (ChessPiece) dragStartSquare
                                .getPiece();
                        if (piece.isPawn()
                                && ((piece.isBlack() && endSquare
                                        .getRankAsChar() == '1') || (!piece
                                        .isBlack() && endSquare.getRankAsChar() == '8'))) {
                            int promoIndex = 0;
                            Object[] message = { "Queen", "Rook", "Knight",
                                    "Bishop" };
                            int choice = JOptionPane.showOptionDialog(this,
                                    "Choisisez une piece : ", "Promotion...",
                                    JOptionPane.YES_NO_OPTION,
                                    JOptionPane.QUESTION_MESSAGE, null,
                                    message, null);
                            switch (choice) {
                            case 0:
                                promoIndex = Queen.INDEX;
                                break;
                            case 1:
                                promoIndex = Rook.INDEX;
                                break;
                            case 2:
                                promoIndex = Knight.INDEX;
                                break;
                            case 3:
                                promoIndex = Bishop.INDEX;
                                break;
                            }
                            move = new ChessMove(board, dragStartSquare.getX(),
                                    dragStartSquare.getY(), endSquare.getX(),
                                    endSquare.getY(), promoIndex);
                        } else {
                            move = new ChessMove(board, dragStartSquare.getX(),
                                    dragStartSquare.getY(), endSquare.getX(),
                                    endSquare.getY());
                        }
                        board.playMove(move);

                        //SI il y a eu uue exception, on n'arrive jamais ici
                        Maillon toAdd = linker.createMaillon(fen, san, san
                                .moveToString(move), currentFen, page);

                        linker.addFEN(currentFen, toAdd);

                        if (!page.isFirstMoveSet()) {
                            page.setFirstMaillon(toAdd);
                        }

                        currentFen = fen.boardToString(board);

                        boolean samePage = toAdd.getPage() == page;

                        reloadPage(getPage(), currentFen, samePage);
                        if (!samePage) {
                            reloadPage(toAdd.getPage(), currentFen);
                        }
                        //Insrer le newMove juste apres le selected en tant
                        // que variation de niveau courant + 1
                        // SI le coup n'a pas de successeurs et SI le coup ne
                        // fait pas changer de variation
                    } catch (IllegalMoveException e1) {
                        System.out.println("Coup Illegal");
                    } catch (IOException e1) {
                        e1.printStackTrace();
                    } catch (AmbiguousChessMoveException e1) {
                        e1.printStackTrace();
                    } finally {
                        boardView.setFen(currentFen);
                        dragStartSquare = null;
                    }

                } else {
                    dragStartSquare = null;
                }
            }

        }

        /**
         * 
         * @see java.awt.event.MouseListener#mouseEntered(java.awt.event.MouseEvent)
         */
        public void mouseEntered(MouseEvent e) {
        }

        /*
         * (non-Javadoc)
         * 
         * @see java.awt.event.MouseListener#mouseExited(java.awt.event.MouseEvent)
         */
        public void mouseExited(MouseEvent e) {
        }

        /**
         * 
         * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
         */
        public void actionPerformed(ActionEvent e) {
            if (e.getSource() == startButton) {
                selectFirstMoveView();
            } else if (e.getSource() == backwardButton) {
                selectPrevMoveView();
            } else if (e.getSource() == forwardButton) {
                selectNextMoveView();
            } else if (e.getSource() == endButton) {
                selectLastMoveView();
            }
        }

        /**
         *  
         */
        private void selectFirstMoveView() {
            while (treeListIterator.hasPrevious()) {
                treeListIterator.previous();
            }
            if (treeListIterator.hasNext()) {
                setSelectedMoveView(treeListIterator.next());
            }
        }

        /**
         *  
         */
        private void selectLastMoveView() {
            while (treeListIterator.hasNext()) {
                treeListIterator.next();
            }
            if (treeListIterator.hasPrevious()) {
                setSelectedMoveView(treeListIterator.previous());
            }
        }

        /**
         *  
         */
        private void selectNextMoveView() {
            //LATER demander quelle branche on choisit
            if (treeListIterator.hasNext()) {
                setSelectedMoveView(treeListIterator.next());
            }
        }

        /**
         *  
         */
        private void selectPrevMoveView() {
            if (treeListIterator.hasPrevious()) {
                treeListIterator.previous();
                if (treeListIterator.hasPrevious()) {
                    setSelectedMoveView(treeListIterator.previous());
                }
            }
        }

        public Page getPage() {
            return page;
        }
    }
}

