/*
Pon : an Opening Notepad
Copyright (C) 2005 Pierre Trocm

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

package pon;

import ictk.boardgame.Board;
import ictk.boardgame.IllegalMoveException;
import ictk.boardgame.Move;
import ictk.boardgame.OutOfTurnException;
import ictk.boardgame.chess.AmbiguousChessMoveException;
import ictk.boardgame.chess.io.FEN;
import ictk.boardgame.chess.io.SAN;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Stack;

/*
 * Created on 30 nov. 2004
 */

/**
 * @author Pierre Trocm
 * @version 0.8 - 28 Dcembre 2004
 */
public class Linker implements Serializable {

    private static final long serialVersionUID = 1L;

    private Map<String, List<Maillon>> fen2Maillon;

    private Chapitre sommaire;

    private String date;

    public Linker() {
        this(new Chapitre("Sommaire") {
            private static final long serialVersionUID = 1L;

            /**
             * Permet de s'assurer qu'on ne dplace pas le sommaire
             */
            public void setChapitre(Chapitre chapitre) {
                return;
            }
        });
    }

    public Linker(Chapitre sommaire) {
        fen2Maillon = new Hashtable<String, List<Maillon>>();
        this.sommaire = sommaire;
    }

    /**
     * @return Returns the sommaire.
     */
    public Chapitre getSommaire() {
        return sommaire;
    }

    /**
     * @param sommaire
     *            The sommaire to set.
     */
    public void setSommaire(Chapitre sommaire) {
        this.sommaire = sommaire;
    }

    /**
     * Ajoute un lien maillon/FEN.
     * 
     * @param fen
     *            la fen  ajouter
     * @param linked
     *            le maillon  ajouter
     * @return <code>true</code> si ce couple fen/Maillon n'tait pas dj
     *         prsent, <code>false</code> sinon.
     */
    public boolean addFEN(String longFen, Maillon linked) {
        String fen = reduxFen(longFen);
        List<Maillon> set;

        if (!fen2Maillon.containsKey(fen)) {
            set = new ArrayList<Maillon>();
            set.add(linked);
        } else {
            //chercher si le coups n'esp as dja prsent
            set = fen2Maillon.get(fen);
            for (Maillon maillon : set) {
                if (maillon.getMove().equals(linked.getMove())) {
                    return false;
                    //RIen  faire,,le coup et la positions sont dja prsents.
                }
            }
            set.add(linked);
        }
        fen2Maillon.put(fen, set);
        return true;
    }

    /**
     * Transforme une fen en fen Rduite. Le nmro du coup est ot, ainsi que le
     * nombre de coups sans pions jous.
     * 
     * @param fen
     *            la fen  recourcir
     * @return la fen rduite.
     */
    private String reduxFen(String fen) {
        String toReturn = new String();
        String[] splittedFen = fen.split(" ");
        for (int i = 0; i < 4; i++) {
            toReturn += splittedFen[i] + (i == 3 ? "" : " ");
        }
        return toReturn;
    }

    public List<Maillon> getMaillonsElementsFrom(String fen) {
        return fen2Maillon.get(reduxFen(fen));
    }

    public Iterator<String> fenIterator() {
        return fen2Maillon.keySet().iterator();
    }

    public static Linker loadSorFile(String fileName) {
        return loadSorFile(new File(fileName));
    }

    public static Linker loadSorFile(File file) {
        String header = "@ Scid opening repertoire file.  Updated: ";
        Linker linker = new Linker();
        Chapitre sommaire = linker.getSommaire();
        Stack<Chapitre> chapterStack = new Stack<Chapitre>();
        Stack<String> fenStack = new Stack<String>();

        chapterStack.add(sommaire);
        fenStack.add(Page.StartFEN);
        FileReader fileReader = null;
        BufferedReader reader = null;
        Page currentPage = null;
        String commentToSet = new String();
        int pageCount = 0;
        boolean jouee = true;
        boolean chapterInit = false;
        FEN fen = new FEN();

        try {

            fileReader = new FileReader(file);
            reader = new BufferedReader(fileReader);
            String firstLine = reader.readLine();

            if (!firstLine.startsWith(header)) {
                return null;
            }
            //Initialisation
            String date = firstLine.substring(header.length() - 1);
            linker.setDate(date);

            while (reader.ready()) {

                String readLine = reader.readLine();

                readLine = readLine.trim();

                if (readLine.length() == 0) {
                    continue;
                }

                switch (readLine.charAt(0)) {
                case '#':
                    commentToSet = readLine.substring(1);
                    break;
                case '[':
                    chapterInit = true;
                    Chapitre toAdd;
                    try {
                        toAdd = new Chapitre(readLine.split(";")[1]);
                    } catch (ArrayIndexOutOfBoundsException e) {
                        toAdd = new Chapitre();
                    }
                    chapterStack.peek().add(toAdd);
                    chapterStack.add(toAdd);
                case '-':
                    jouee = chapterInit;
                //SI on initialise un chapter, on le joue
                case '+':
                    String title;
                    //+ ou -

                    //INVARIANT
                    String[] splittedLine = readLine.split(";");
                    String line = splittedLine[0];
                    try {
                        title = splittedLine[1].trim();
                    } catch (ArrayIndexOutOfBoundsException e) {
                        title = "Page " + ++pageCount;
                    }

                    /*
                     * On est en situation o dans line il y a des coups, cas du
                     * prfixe + ou -
                     */
                    try {
                        currentPage = new Page(title, null, jouee);
                        currentPage.setPrefixFen(fenStack.peek());
                        currentPage.setComments(commentToSet);
                        commentToSet = new String();

                        Board board = fen.stringToBoard(fenStack.peek());

                        SAN san = new SAN();

                        for (String currentToken : line.substring(1).split(
                                "[ ]*[0-9]+\\.")) {
                            for (String moveToken : currentToken.split(" ")) {

                                if (moveToken.length() == 0) {
                                    continue;
                                }

                                String fromFen = fen.boardToString(board);
                                Maillon currentMaillon = linker.createMaillon(
                                        fen, san, moveToken.trim(), fromFen,
                                        currentPage);
                                boolean isNovelty = linker.addFEN(fromFen,
                                        currentMaillon);

                                if (currentPage.getFen() == null) {
                                    if (isNovelty) {
                                        currentPage.setFen(fromFen);
                                        currentPage
                                                .setFirstMaillon(currentMaillon);
                                        chapterStack.peek().add(currentPage);
                                    } else {
                                        currentPage.addToPrefix(currentMaillon);
                                    }
                                } else {
                                    currentPage = currentMaillon.getPage();
                                }

                                board.playMove(san.stringToMove(board,
                                        moveToken.trim()));
                            }
                        }
                        if (chapterInit) {
                            fenStack.add(fen.boardToString(board));
                            chapterInit = false;
                        }

                    } catch (OutOfTurnException e) {
                        e.printStackTrace();
                    } catch (AmbiguousChessMoveException e) {
                        e.printStackTrace();
                    } catch (IllegalMoveException e) {
                        e.printStackTrace();
                    }

                    break;
                case ']':
                    chapterStack.pop();
                    fenStack.pop();
                    break;
                }

            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            try {
                try {
                    reader.close();
                } catch (IOException e1) {
                    e1.printStackTrace();
                }
                try {
                    fileReader.close();
                } catch (IOException e2) {
                    e2.printStackTrace();
                }
            } catch (NullPointerException e) {
                e.printStackTrace();
            }
        }

        return linker;
    }

    public Maillon createMaillon(FEN fen, SAN san, String moveString,
            String fromFen, Page containingPage) throws IOException,
            AmbiguousChessMoveException, IllegalMoveException {

        Board myBoard = fen.stringToBoard(fromFen);
        Move move = san.stringToMove(myBoard, moveString);

        myBoard.playMove(move);

        String resultingFen = fen.boardToString(myBoard);

        if (fen2Maillon.keySet().contains(reduxFen(fromFen))) {
            //SI la fen de base est dja contenue
        }

        if (fen2Maillon.keySet().contains(reduxFen(resultingFen))) {
            //SI la fen d'arrive est dja contenue
            containingPage = getMainLineFrom(reduxFen(resultingFen)).getPage();
        }

        return new Maillon(moveString, containingPage);
    }

    public Maillon getMainLineFrom(String fen) {
        try {
            return fen2Maillon.get(reduxFen(fen)).get(0);
        } catch (NullPointerException e) {
            return null;
        }
    }

    public List<Maillon> getVariationsFrom(String fen) {
        try {
            List<Maillon> list = fen2Maillon.get(reduxFen(fen));
            return list.subList(1, list.size());
        } catch (NullPointerException e) {
            return Collections.emptyList();
        }
    }

    public void save(String filename) {
        save(new File(filename));
    }

    /**
     * @param file
     */
    public void save(File file) {
        try {
            if (!file.getName().endsWith(".pon")) {
                file = new File(file.getName() + ".pon");
            }
            FileOutputStream fos = new FileOutputStream(file);
            ObjectOutputStream oos = new ObjectOutputStream(fos);
            oos.writeObject(this);
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public static Linker load(String fileName) throws IOException,
            ClassNotFoundException {
        return load(new File(fileName));
    }

    /**
     * @param file
     * @return
     * @throws IOException
     * @throws ClassNotFoundException
     */
    public static Linker load(File file) throws IOException,
            ClassNotFoundException {
        FileInputStream fis = new FileInputStream(file);
        ObjectInputStream ois = new ObjectInputStream(fis);
        return (Linker) ois.readObject();
    }

    /**
     * @return Returns the date.
     */
    public String getDate() {
        return date;
    }

    /**
     * @param date
     *            The date to set.
     */
    public void setDate(String date) {
        this.date = date;
    }

    /**
     * @param fromFen
     * @param toDelete
     * @param page
     */
    public void deleteMaillon(String fromFen, Maillon toDelete, Page page) {
        //LATER une void maillon.delete() pour s'assurer que le maillon est bien supprim
        try {
            SAN san = new SAN();
            List<Maillon> toModify = fen2Maillon.get(reduxFen(fromFen));

            if (toModify != null) {
                toModify.remove(toDelete);
                
                if (toModify.isEmpty()) {
                    fen2Maillon.remove(reduxFen(fromFen));
                }

                //Appliquer la rccurence aux lments suivants :
                Board board = FENHandler.getBoard(fromFen);
                board.playMove(san.stringToMove(board, toDelete.getMove()));
                fromFen = FENHandler.getFen(board);
                
                Maillon[] succeseurs = new Maillon[1];
                getMaillonsElementsFrom(fromFen).toArray(succeseurs);

                for (Maillon maillon : succeseurs) {
                    if (maillon != null && maillon.getPage() == page) {
                        deleteMaillon(fromFen, maillon, page);
                        //LATER trouver un moyen de vrifier qu'il n'y a pas d'lments "flottants"
                        //Aux embranchements de variations
                    }
                }
            }
        } catch (IOException e) {
            e.printStackTrace();
        } catch (OutOfTurnException e) {
            e.printStackTrace();
        } catch (AmbiguousChessMoveException e) {
            e.printStackTrace();
        } catch (IllegalMoveException e) {
            e.printStackTrace();
        }
    }
}